import java.util.*;

/**
 * This class is used to generate random Q-matrices. The matrix generated could
 * be either symmetric or non-symmetric
 *
 * @author Vivek Jayaswal
 * 
 *	-	Modified the generated Q-matrices to ensure that all the elements of the
 *		Q-matrix add up to 1	 	
 */

public class QMatrixGenerator {
	
	double[][] qMatrix = new double[4][4];
	Random r = new Random();
		
	/**	This method is called to generate Q-matrices with known probability 
	 *  distribution. The main diagonal elements are much bigger than the 
	 *	off-diagonal elements 
	 *
	 *	@param	pA			Probability of nucleotide A
	 *	@param	pC			Probability of nucleotide C
	 *	@param	pG			Probability of nucleotide G
	 *	@param	pT			Probability of nucleotide T
	 *	@param 	option		1 -> Generate only symmetric matrices
	 *						2 -> Randomly generate symmetric/ non-symmetric matrices
	 */
	double[][] generateMatrix(double pA, double pC, double pG, double pT, int option) {
	
		double d = r.nextDouble();
		
		double sum = pA + pC + pG + pT;
		
		//System.out.println("Sum of probabilities = " + sum);
		
		//if(d > 0.5)  {
		if(option == 1) {
			getSymmetricMatrix(pA, pC, pG, pT);
		}	
		else {
			if(d > 0.5)  {
				getNonSymmetricMatrix(pA, pC, pG, pT);
			}
			else {
				getSymmetricMatrix(pA, pC, pG, pT);
			}	
		}
		
		return qMatrix;
	}	
	
	/**	This method is called to generate random Q-matrices with unknown
	 *	probability distribution. The main diagonal elements are much bigger 
	 *	than the off-diagonal elements 
	 *
 	 *	@param	pA			Probability of nucleotide A
	 *	@param	pC			Probability of nucleotide C
	 *	@param	pG			Probability of nucleotide G
	 *	@param	pT			Probability of nucleotide T
	 */
	double[][] generateMatrix() {
	
		double d = r.nextDouble();

  		double pA = r.nextDouble();
  		double pC = r.nextDouble();
  		double pG = r.nextDouble();
  		double pT = r.nextDouble();
  		double sum = pA + pC + pG + pT;
  		
  		pA /= sum;
  		pC /= sum;
  		pG /= sum;
  		pT /= sum;
	  				
		if(d > 0.5) {
			getSymmetricMatrix(pA, pC, pG, pT);
		}	
		else {
			getNonSymmetricMatrix(pA, pC, pG, pT);
		}
		
		return qMatrix;
	}	
	
	/**	This method is used to create non-symmetric matrix such that the row
	 *	values add up to P(A), P(C), P(G), P(T)
	 *
	 *	@param	pA			Probability of nucleotide A
	 *	@param	pC			Probability of nucleotide C
	 *	@param	pG			Probability of nucleotide G
	 *	@param	pT			Probability of nucleotide T
	 */
			
	void getNonSymmetricMatrix(double pA, double pC, double pG, double pT) {
		int i;
		int j;
		double sum;
	
		//System.out.println("Non-symmetrix Matrix");
			
		for(i=0; i<4; i++) {
			sum = r.nextDouble();
			if(sum<0.5) sum += 0.5;

			//For each row, > 50% of the probability value should lie on the main
			//diagonal element
			if(i == 0) sum *= pA;	
			else if(i == 1) sum *= pC;	
			else if(i==2) sum *= pG;	
			else if(i==3) sum *= pT;	

			for(j=0; j<4; j++) {
				if(i==j) {
					qMatrix[i][j] = sum;
				}						
				else {
					switch(i) {
						case 0: qMatrix[i][j] = 1/3.0 * (pA - sum); break;
						case 1: qMatrix[i][j] = 1/3.0 * (pC - sum); break;
						case 2: qMatrix[i][j] = 1/3.0 * (pG - sum); break;
						case 3: qMatrix[i][j] = 1/3.0 * (pT - sum); 
					} // end of switch statement
				}
				
			} // end of inner loop
		} // end of outer loop
		
		//Ensure that each row sum adds to the relevant probability value
		for(i=0; i<4; i++) {
			sum = qMatrix[i][0] + qMatrix[i][1] + qMatrix[i][2] + qMatrix[i][3]; 
			
			switch(i) {
				case 0: sum -= pA; break;
				case 1: sum -= pC; break;
				case 2: sum -= pG; break;
				case 3: sum -= pT;
			}
			
			qMatrix[i][i] -= sum;
		}
				
	}
	
	/**	This method is used to create a symmetric matrix such that the row
	 *	values add up to P(A), P(C), P(G), P(T)
	 *
	 *	@param	pA			Probability of nucleotide A
	 *	@param	pC			Probability of nucleotide C
	 *	@param	pG			Probability of nucleotide G
	 *	@param	pT			Probability of nucleotide T
	 */
	void getSymmetricMatrix(double pA, double pC, double pG, double pT) {


		/* Symmetric Matrix with known marginal probabilities. Arrange the 
		 * probabilities in descending order such that a matrix of the form 
		 * 		A1	b	c	d	= X1
		 *		b	A2	c	d	= X2
		 *		c	c	A3	d	= X3
		 *		d	d	d	A4	= X4
		 * can be created wher Xi's represent the row sums. Now, obtain the 
		 * values of b, c, d and A1, A2, A3 , A4
		 * Next, rearrange the matrix such that rows correspond to pA, pC, pG 
		 * and pT 
		 */
		double a1, a2, a3, a4;
		double b, c, d;
		
		System.out.println("Symmetrix Matrix");
		
		//Arrange in descending order of probabilities
		TreeMap tMap = new TreeMap();
		
		tMap.put(new Double(pA), "A ");
		tMap.put(new Double(pC), "C ");
		tMap.put(new Double(pG), "G ");
		tMap.put(new Double(pT), "T ");
		
		// Determine the values - b, c, d, A1, A2, A3, A4
		Set set = tMap.entrySet();
		Iterator itr = set.iterator();
		Map.Entry me;
		String baseValue;
		double row[] = new double[4];
		int sequence[] = new int[4];
		int count = 0;
		
		//Store the probability values in an ascending order					
		while(itr.hasNext()) {
			me = (Map.Entry) itr.next();
			row[count]  = ((Double)me.getKey()).doubleValue();
			
			baseValue = (String)me.getValue();
			
			switch(baseValue.charAt(0)) {
				case 'A': sequence[count++] = 0;break;
				case 'C': sequence[count++] = 1;break;
				case 'G': sequence[count++] = 2;break;
				case 'T': sequence[count++] = 3;
			}
		}
										
		a4 = r.nextDouble();
		if(a4 < 0.5) a4 += 0.5;
		a4 = a4 * row[0];
		
		//while(a4 >= row[0]) a4 -= row[0];		
		d = (row[0] - a4)/3;
		
		//Calculate a3, c
		a3 = r.nextDouble();
		if(a3 < 0.5) a3 += 0.5;
		a3 = a3 * (row[1] - d);
		
		//while(a3 >= (row[1] - d)) a3 = a3 - (row[1] - d);				
		c = (row[1] - a3 - d)/2;
		
		//Calculate a2, b
		a2 = r.nextDouble();					
		
		if(a2 < 0.5) a2 += 0.5;
		a2 = a2 * (row[2] - d - c);

		//while((a2+c+d) >= row[2]) a2 = a2 - (row[2] - d - c);				
		b = row[2] - (a2+d+c);
		
		//Calculate a1
		a1 = r.nextDouble();				
		
		if(a1 < 0.5) a1 += 0.5;
		a1 = a1 * (row[3] - d - c - b);
		
		//while((a1+b+c+d) >= row[3]) a1 = a1 - (row[3] -d -c -b);		
		//double diff = row[3] - (a1+b+c+d);
		//a1 += diff;
		
		//Re-arrange the rows in the correct order A, C, G, T
		for(int i=0; i<4; i++) {
			qMatrix[sequence[0]][i] = d;
			qMatrix[sequence[1]][i] = c;
			qMatrix[sequence[2]][i] = b;
			qMatrix[sequence[3]][i] = b;
		}
		
		//Elements along the main diagonal
		qMatrix[sequence[0]][sequence[0]] = a4;
		qMatrix[sequence[1]][sequence[1]] = a3;
		qMatrix[sequence[2]][sequence[2]] = a2;
		qMatrix[sequence[3]][sequence[3]] = a1;		
		
		//Row containing c c d
		qMatrix[sequence[1]][sequence[0]] = d;
		
		//Row containing b c d a2
		qMatrix[sequence[2]][sequence[0]] = d;
		qMatrix[sequence[2]][sequence[1]] = c;
		
		//Row containing b c d a1
		qMatrix[sequence[3]][sequence[0]] = d;
		qMatrix[sequence[3]][sequence[1]] = c;
		
		//For each row, > 50% of the probability value should lie on the main
		//diagonal element		
		double sum;
		for(int i=0; i<4; i++) {
			sum = qMatrix[i][0] + qMatrix[i][1] + qMatrix[i][2] + qMatrix[i][3]; 
			
			switch(i) {
				case 0: sum -= pA; break;
				case 1: sum -= pC; break;
				case 2: sum -= pG; break;
				case 3: sum -= pT;
			}
			
			qMatrix[i][i] -= sum;
		}
				
	}
		
	// This method is used to display the Q-matrix created on the console		
	void displayMatrix() {

		for(int i=0; i<4; i++) {
			for(int j=0; j<4; j++) {
				System.out.print(qMatrix[i][j] + "\t");
			}
			System.out.println();
		}
	
	}
	
	public static void main (String args[]) {
		
		Random r1 = new Random();
		
		double pA = r1.nextDouble(); //0.15232321;
		double pC = r1.nextDouble(); //0.258642966;
		double pG = r1.nextDouble(); //0.298593513;		
		double pT = r1.nextDouble(); //1 - (pA + pC + pG);
		
		double sum = pA + pC + pG + pT;

/*		
		System.out.println("P Values Selected ");
		System.out.println("PA = " + pA);
		System.out.println("PC = " + pC);
		System.out.println("PG = " + pG);
		System.out.println("PT = " + pT);	
		
		System.out.println("P Values Normalized");
		System.out.println("PA = " + pA/sum);
		System.out.println("PC = " + pC/sum);
		System.out.println("PG = " + pG/sum);
		System.out.println("PT = " + pT/sum);	
*/
		QMatrixGenerator qMatGen = new QMatrixGenerator();		
		qMatGen.generateMatrix(pA/sum, pC/sum, pG/sum, pT/sum, 1);	
					
		qMatGen.displayMatrix();
	}
}
