/*
 * The author of this software is Catherine Loader, c@herine.net
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose without fee is hereby granted, with the exceptions noted below,
 * and provided that this entire notice is included in all copies of any
 * software which is or includes a copy or modification of this software
 * and in all copies of the supporting documentation for such software.
 * THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTY.  IN PARTICULAR, NEITHER THE AUTHOR NOR LUCENT TECHNOLOGIES
 * MAKE ANY REPRESENTATION OR WARRANTY OF ANY KIND CONCERNING THE
 * MERCHANTABILITY OF THIS SOFTWARE OR ITS FITNESS FOR ANY PARTICULAR PURPOSE.
 */
/*
 * This file includes the `raw' functions for evaluating binomial and
 * Poisson probabilities. These underly all the other probability
 * computations.
 *
 * double dbinom_raw(x,n,p,q,give_log)
 * double dpois_raw(x,lambda,give_log)
 *   
 * arguments x and n have type NTYPE. This can be defined either to be
 *   integer or double in the dbinom.h header file.
 * p, q and lambda have type double.
 * if give_log = 0, the probability is returned. if give_log=1, the
 *   log of the probability is returned.
 *
 * Notes:
 * (1) dbinom has both p and q arguments. This is useful when one may be
 *     represented more accurately than the other (in particular, in df()).
 * (2) there is no check that x, n are integers - the routines do not require
 *     this. Also, there is no check for validity of parameters p, q, lambda
 *     - do this in the calling function as appropriate.
 */

#include <math.h>
#include "dbinom.h"

double dbinom_raw(x,n,p,q,give_log)
NTYPE x, n;
double p, q;
int give_log;
{ double f, lc;

  if (p==0.0) return((x==0) ? D_1 : D_0);
  if (q==0.0) return((x==n) ? D_1 : D_0);

  if (x==0)
  { lc = (p<0.1) ? -bd0(n,n*q) - n*p : n*log(q);
    return( DEXP(lc) );
  }

  if (x==n)
  { lc = (q<0.1) ? -bd0(n,n*p) - n*q : n*log(p);
    return( DEXP(lc) );
  }

  if ((x<0) | (x>n)) return( D_0 );

  lc = STIRLERR(n) - STIRLERR(x) - STIRLERR(n-x)
         - bd0(x,n*p) - bd0(n-x,n*q);
  f = (PIx2*x*(n-x))/n;

  return( FEXP(f,lc) );
}

double dpois_raw(x,lambda,give_log)
NTYPE x;
int give_log;
double lambda;
{
  if (lambda==0) return( (x==0) ? D_1 : D_0 );
  if (x==0) return( DEXP(-lambda) );
  if (x<0) return( D_0 );

  return(FEXP( PIx2*x, -STIRLERR(x)-bd0(x,lambda) ));
}
