/*
  Generate MC images of the input sets using either a Markov model or boostrap.
  Bootstrap can protect sites. This is particularly important for longer motifs
  that could easily be chopped if the block size is small relative to the width.
*/


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <limits.h>
#include <float.h>
#include <assert.h>
#include "my_types.h"
#include "data_interface.h"
#include "misc_functions.h"
#include "markov.h"
#include "pwms.h"
#include "sitePrtctdBtstrp.h"
#include "mc_sets.h"



// Markov model parameters (relevant only if markov random model is used)
static int markovOrder, trainFileScanMode;
static double pseudoCount;

static enum MCmodelType mc_model; // which MC model are we using

gappedSeqVec gen_mc_ds(gappedSeqVec moldSet, RandomDsCreatorType *rdc, motif_Struct motif)	{
 	// given a dataset and a motif return an MC image of the data part
    // motif is only used if bootProtectsSites is set
	int totLen = gappedSeqVecTotalLen(moldSet);
	iLetterVec randChunk = gen_MCblock(rdc, motif, totLen);
	gappedSeqVec mc_set = iLetterVec2gappedSeqVec(randChunk, moldSet);
	FREE_ATOMS_VEC(randChunk);
	return mc_set;
}

/*
chunksSeqVec gen_mc_ds(chunksSeqVec moldSet, RandomDsCreatorType *rdc, motif_Struct motif)
     // given a dataset and a motif return an MC image of the data part
     // motif is only used if bootProtectsSites is set
{
  chunksSeqVec mc_set;
  int totLen;
  iLetterVec randChunk;

  totLen = chunkSeqVecTotalLen(moldSet);
  randChunk = gen_MCblock(rdc, motif, totLen);  
  mc_set = iLetterVec2chunksSeqVec(randChunk, moldSet);// reshape randChunk to an MC set image of *set
  FREE_ATOMS_VEC(randChunk);

  return mc_set;
}
*/

iLetterVec gen_MCblock(RandomDsCreatorType *rdc, motif_Struct motif, int len)
     // given a dataset and a motif return an MC block of length len
     // motif is only used if bootProtectsSites is set
{
  iLetterVec randChunk;

  if (mc_model == Bootstrap) { // bootstrap
    if (rdc->bootBlocks.motifID != motif.id || rdc->bootBlocks.len == 0) // do we need to construct the blocks?
       gen_bootBlocksDB(rdc->trainData, motif, &(rdc->bootBlocks));
    randChunk = gen_ranBootBlock(rdc->trainData, rdc->bootBlocks, len);
  }
  else // Markov
    randChunk = genMarkovBlock(rdc->markovModel, len);

  return randChunk;
}


void set_mcModel(enum MCmodelType mc_modelValue, int markovOrderValue, double pseudoCountValue, int trainFileScanModeValue)
//       Sets the MC model that will be used to generate random sets
{
  mc_model = mc_modelValue;
  markovOrder = markovOrderValue;
  pseudoCount = pseudoCountValue;
  trainFileScanMode = trainFileScanModeValue;
}


RandomDsCreatorType *alloc_RandomDsCreator(iLetterVec trainData)
     // Returns a pointer to an allocated structure
{
  RandomDsCreatorType *randDsCreator;

  assert( randDsCreator = (void *) calloc(1, sizeof(RandomDsCreatorType)) );

  randDsCreator->trainData = trainData;
  randDsCreator->randModel = mc_model;
  if (mc_model == Markov)
    randDsCreator->markovModel = trainMarkovModel(randDsCreator->trainData, markovOrder, pseudoCount, trainFileScanMode);
  else
    randDsCreator->bootBlocks.motifID = -1; // make sure these will be recalculated on the fly if Bootstrap

  return randDsCreator;
}


void free_RandomDsCreatorType(RandomDsCreatorType *rdc)
		// frees the allocated arrays
{
	FREE_ATOMS_VEC(rdc->trainData);
	freeMarkovModel(rdc->markovModel);
	free_bootBlocks(&(rdc->bootBlocks));
}
