#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <errno.h>
#include <string.h>
#include <math.h>
#include "my_types.h"
#include "data_interface.h"
#include "misc_functions.h"
#include "rand.h"


int *rand_perm_prealloc(int *perm, int n)
     /*
       Returns a random permutation in S_n w/o worrying about memory  allocation
     */
{
        int i, pos, temp;

        for (i = 0; i < n; i++)
                perm[i] = i+1;

        for (i = 0; i < n-1; i++) {
                pos = INT_RAND % (n-i);
                temp = perm[n-i-1];
                perm[n-i-1] = perm[pos];
                perm[pos] = temp;
        }
        return perm;
}


intVec multiset_multiplicities(realVec mset)
     /*
       Given a real valued multiset mset the function returns an intVec of the multiplicities of the sorted mset
     */
{
  intVec multiplicities;
  int num_matches = 1;
  intVecEntry *multips;
  int nElems=0, i;

  assert(multips = (intVecEntry *) calloc(mset.len, sizeof(intVecEntry)));
  qsort((void *)mset.entry, (size_t) mset.len, sizeof(realVecEntry), (void *)realVecEntryComp);
  for (i = 0; i < mset.len-1; i++) {
    if (mset.entry[i] == mset.entry[i+1])
      num_matches++;
    else {
      multips[nElems++] = num_matches; // for the mw_lattice function
      num_matches = 1;
    }
  }
  multips[nElems++] = num_matches;
  multiplicities.len = nElems;
  multiplicities.entry = my_realloc(multips, sizeof(intVecEntry)*nElems);
  return multiplicities;
}


double myNaN()
     // Returns a Nan. For some reason the NAN macro isn't doing that
{
  double r=0, q=0;

  return r/q;
}


int multinomialDraw(double *pmf, int nItems)
     /* 
	A somewhat inefficient draw from a multinomial distribution with n items.
	Possible improvements: sort the pmf in decreasing order (keeping the sorting permutation)
	Sum the probabilities as you sort.
     */
{
  int i;
  double r, p=0;

  r = FLOAT_RAND;
  i = 0;
  do
    p += pmf[i++];
  while (i < nItems && r > p);
  return i-1;
}


double compMean(realVec sample)
     // Return the mean of sample
{
  int i;
  double sum=0;

  for (i = 0; i < sample.len; i++)
    sum += sample.entry[i];

  return sum / sample.len;
}


double compStd(realVec sample)
     // Return the standard deviation of sample
{
  int i;
  double dev=0, mean;

  mean = compMean(sample);
  for (i = 0; i < sample.len; i++)
    dev += (sample.entry[i]-mean) * (sample.entry[i]-mean);

  return sqrt(dev / sample.len);
}


void errorMessage(char *message)
     // Print the message and abort
{
  printf("\nAborting: %s\n", message);
  exit(0);
}


long my_atol(char *string)
     // Execute an error alert atol()
{
  long n;

  errno = 0;
  n = strtol(string, (char **)NULL, 10);
  if (errno)
    ERROR(("Cannot convert %s to an integer", string))
  return n;
}


double my_atod(char *string)
     // Execute an error alert atol()
{
  double x;

  errno = 0;
  x = strtod(string, (char **)NULL);
  if (errno)
    ERROR(("Cannot convert %s to a real number", string))
  return x;
}


void *my_realloc(void *ptr, size_t size)
     // assert(...realloc(...) ) is dangerous if size=0
{
  void *new_ptr;

  if ( ((new_ptr = realloc(ptr, size)) == NULL)  &&  (size != 0) )
    ERROR(("\nCannot realloc in my_realloc"))
  return new_ptr;
}


void checkRangeD(double x, double minX, double maxX, char *nameX)
     // Aborts if x is not in the closed interval [minX, maxX]
{
  if (x < minX || x > maxX)
    ERROR(("At %f %s is not in [%f,%f]", x, nameX, minX, maxX))
}


void checkRangeI(int x, int minX, int maxX, char *nameX)
     // Aborts if x is not in the closed interval [minX, maxX]
{
  if (x < minX || x > maxX)
    ERROR(("At %d %s is not in [%d,%d]", x, nameX, minX, maxX))
}


void setTRUE(Boolean *p, int from, int len) {
  int i;
  for (i = from; i < from + len; i++)
    p[i] = TRUE;
}


int findStringOnce(char *string, stringVec strings)
     // Returns the index of first exact occurence of string in strings or aborts if not there
{
  intVec parmIndex;
  int ind;

  parmIndex = findStringInStringVec(string, strings);
  if (parmIndex.len < 1)
    ERROR(("%s is not found in my list", string))
  else {
    ind = parmIndex.entry[0];
    free(parmIndex.entry);
    return ind;
  }
}


int findString(char *string, stringVec strings)
     // Returns the index of the exact occurence of string in strings or -1 if none.
{
  intVec parmIndex;
  int ind;

  parmIndex = findStringInStringVec(string, strings);
  if (parmIndex.len < 1)
    return -1;
  else {
    ind = parmIndex.entry[0];
    free(parmIndex.entry);
    return ind;
  }
}


intVec findStringInStringVec(char *string, stringVec strings)
      // Returns all indices i for which strings[i]==string (exact complete matches only)
{
  int i, n=0;
  intVec locs;

  locs = alloc_intVec(strings.len);
  for (i = 0; i < strings.len; i++)
    if (strcmp(string, strings.entry[i]) == 0)
      locs.entry[n++] = i;
  realloc_intVec(&locs, n);

  return locs;
}


char *allocAndStrCpy(char *str)
     // strcpy + malloc
{
  char *copy;

  assert( copy = (void *) malloc((1+strlen(str))*sizeof(char)) );
  strcpy(copy, str);
  return copy;
}


char *allocAndStrCat(char *str1, char *str2)
     // returns a string which is the concatenation of str2 to str1
{
  char *cat;

  assert( cat = (void *) malloc((1+strlen(str1)+strlen(str2))*sizeof(char)) );
  strcpy(cat, str1);
  strcat(cat, str2);
  return cat;
}


double *allocAndSetD(double x)
     // Allocates a double and sets it to x
{
  double *p;
  assert( p = malloc(sizeof(double)) );
  *p = x;
  return p;
}


iLetterVec getReverseComplement(const iLetterVec seq)
     // Allocate and returns the reverse complement of seq
	 // Changed by Anand: If seq contains values which are not among [0, 4), 
	 // then the returned RC contains the same invalid value as in seq
{
  iLetterVec rcSeq;
  int i;

  rcSeq = alloc_iLetterVec(seq.len);
  for (i = 0; i < seq.len; i++)	{
	if (seq.entry[i] >= 0 && seq.entry[i] < 4)	{
	  rcSeq.entry[seq.len-1-i] = 3 - seq.entry[i];
	}
	else	{
	  rcSeq.entry[seq.len-1-i] = seq.entry[i];
	}
  }
  return rcSeq;
}

// Determines the reverse complement gapped sequence of seq
gappedSeq getSeqReverseComplement(const gappedSeq seq)	{
	gappedSeq rcSeq;
	iLetterVec body = getReverseComplement(seq.body);
	int numChunks = seq.numChunks;
	
	rcSeq.header = NULL;
	rcSeq.header_len = 0;
	rcSeq.body = body;
	rcSeq.numChunks = seq.numChunks;
	rcSeq.chunkLens = calloc(numChunks, sizeof(int));
	rcSeq.chunkOffsets = calloc(numChunks, sizeof(int));
	
	int offset = 0, i;
	for (i = numChunks - 1; i >= 0; i--)	{
		rcSeq.chunkOffsets[numChunks - 1 - i] = offset;
		rcSeq.chunkLens[numChunks - 1 - i] = seq.chunkLens[i];
		int gap = 0;
		if (i > 0)	gap = seq.chunkOffsets[i] - (seq.chunkOffsets[i-1] + seq.chunkLens[i-1]);
		offset += seq.chunkLens[i] + gap;
	}
	return rcSeq;
}
